/*
 * Copyright (c) 2023, Azul Systems, Inc. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

import jdk.crac.Core;
import jdk.internal.perf.Perf;
import jdk.test.lib.crac.CracBuilder;
import jdk.test.lib.crac.CracEngine;
import jdk.test.lib.crac.CracTest;
import jdk.test.lib.crac.CracTestArg;
import jdk.test.lib.Utils;

import java.lang.management.ManagementFactory;
import java.nio.file.Path;
import java.util.*;

import static jdk.test.lib.Asserts.assertLT;
import static jdk.test.lib.Asserts.assertLTE;

/**
 * @test
 * @library /test/lib
 * @modules java.base/jdk.internal.perf:+open
 * @build ResetStartTimeTest
 * @run driver/timeout=60 jdk.test.lib.crac.CracTest false
 * @run driver/timeout=60 jdk.test.lib.crac.CracTest true
 * @requires (os.family == "linux")
 */
public class ResetStartTimeTest implements CracTest {

    @CracTestArg(0)
    boolean resetUptime;

    static private final long WAIT_TIMEOUT = 2 * 1000; // msecs

    @Override
    public void test() throws Exception {
        CracBuilder builder = new CracBuilder().vmOption("--add-opens").vmOption("java.base/jdk.internal.perf=ALL-UNNAMED");
        builder.startCheckpoint().waitForCheckpointed();
        if (!resetUptime) {
            builder.vmOption("-XX:+UnlockDiagnosticVMOptions");
            builder.vmOption("-XX:-CRaCResetStartTime");
        }
        builder.captureOutput(true).doRestore().waitForSuccess()
                .outputAnalyzer().shouldContain(RESTORED_MESSAGE);
    }

    @Override
    public void exec() throws Exception {
        Thread.sleep(WAIT_TIMEOUT);
        final long uptime0 = ManagementFactory.getRuntimeMXBean().getUptime();
        final long counter0 = Perf.getPerf().highResCounter();

        Core.checkpointRestore();
        final long uptime1 = ManagementFactory.getRuntimeMXBean().getUptime();
        final long counter1 = Perf.getPerf().highResCounter();

        if (resetUptime) {
            assertLT(uptime1, uptime0);
            assertLT(uptime1, WAIT_TIMEOUT);
        } else {
            assertLTE(uptime0, uptime1);
        }
        assertLT(counter0, counter1);

        System.out.println(RESTORED_MESSAGE);
    }
}
